#!/usr/bin/perl
use strict;
use warnings;
use Fcntl qw(:flock);
use Encode qw(decode encode);

# ---------------------------------------------------------
# 設定
# ---------------------------------------------------------
my $log_file = "../dat/chat_log.csv";
my $id_file  = "../dat/last_id.txt";
my $max_log  = 30;                  # 保持する最大行数
my $timeout  = 10;                  # 処理タイムアウト(秒)

$SIG{ALRM} = sub { die "timeout" };
alarm($timeout);

# ---------------------------------------------------------
# パラメータ取得
# ---------------------------------------------------------
sub get_parameters {
    my %params;
    my $buffer;
    if ($ENV{'REQUEST_METHOD'} eq 'POST') {
        read(STDIN, $buffer, $ENV{'CONTENT_LENGTH'});
    } else {
        $buffer = $ENV{'QUERY_STRING'};
    }
    my @pairs = split(/&/, $buffer);
    foreach my $pair (@pairs) {
        my ($name, $value) = split(/=/, $pair);
        $value =~ tr/+/ /;
        $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;
        $name  =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;
        $params{$name} = $value;
    }
    return %params;
}

# ---------------------------------------------------------
# サニタイズ（HTMLエスケープ + CSV用カンマ除外）
# ---------------------------------------------------------
sub sanitize {
    my $text = shift;
    return "" unless defined $text;
    $text = decode('utf-8', $text) unless utf8::is_utf8($text);

    $text =~ s/&/&amp;/g;
    $text =~ s/</&lt;/g;
    $text =~ s/>/&gt;/g;
    $text =~ s/"/&quot;/g;
    $text =~ s/'/&#39;/g;
    $text =~ s/,/&#44;/g;   # カンマを実体参照に置換（CSV崩れ防止）
    $text =~ s/\r?\n/ /g;   # 改行をスペースに置換
    $text =~ s/^\s+|\s+$//g;
    return $text;
}

# ---------------------------------------------------------
# メインロジック
# ---------------------------------------------------------
my %params = get_parameters();
my $action = $params{'action'} // '';
my $since  = int($params{'since'}  // 0); # クライアントが持っている最新ID

# HTTPヘッダー出力
print "Content-Type: text/plain; charset=utf-8\n\n";
binmode(STDOUT, ":utf8");

if ($action eq 'send') {
    # --- メッセージ送信処理 ---
    my $name = sanitize($params{'name'});
    my $msg  = sanitize($params{'message'});

    if ($name ne '' && $msg ne '') {
        # 1. ユニークIDの生成 (last_id.txt を更新)
        my $new_id = 0;
        my $ifh;
        if (open($ifh, "+<", $id_file) || open($ifh, ">", $id_file)) {
            flock($ifh, LOCK_EX);
            $new_id = <$ifh> // 0;
            $new_id++;
            seek($ifh, 0, 0);
            truncate($ifh, 0);
            print $ifh $new_id;
            close($ifh);
        }

        # 2. ログファイルへの追記
        my $fh;
        if (open($fh, "+<:utf8", $log_file) || open($fh, ">:utf8", $log_file)) {
            flock($fh, LOCK_EX);
            my @lines = <$fh>;
            
            # ID, 名前, メッセージ, UNIXタイム
            push @lines, "$new_id,$name,$msg," . time() . "\n";
            
            # 最大件数を超えたら古いものを削除
            if (@lines > $max_log) {
                @lines = splice(@lines, -$max_log);
            }
            
            seek($fh, 0, 0);
            truncate($fh, 0);
            print $fh @lines;
            close($fh);
            print "OK";
        } else {
            print "ERROR: FILE_IO";
        }
    } else {
        print "ERROR: EMPTY_DATA";
    }
} else {
    # --- メッセージ取得処理 ---
    if (-e $log_file) {
        if (open(my $fh, "<:utf8", $log_file)) {
            flock($fh, LOCK_SH);
            while (<$fh>) {
                my ($id) = split(/,/, $_);
                # クライアントが持っているIDより大きいものだけを返す
                if (int($id) > $since) {
                    print $_;
                }
            }
            close($fh);
        }
    }
}

alarm(0);
exit;