#!/usr/bin/perl
use strict;
use warnings;
use Fcntl qw(:flock :DEFAULT);

# 設定 -----------------------------
my $count_file = "./dat/count.txt";
my $log_file   = "./dat/access.log";
my $ip_file    = "./dat/ip_limit.txt";
my $limit_sec  = 10;    # 同一IPの無視期間
my $expire_sec = 3600;  # 1時間以上前のIPデータは削除
# ---------------------------------

my $ip   = $ENV{'REMOTE_ADDR'}    || "unknown";
my $ua   = $ENV{'HTTP_USER_AGENT'} || "unknown";
my $time = time;

# --- 1. IP制限の読み込みと掃除 ---
my %last;
# 読み書きモードで開き、ロックする（肥大化と競合を防止）
sysopen(my $fh_ip, $ip_file, O_RDWR | O_CREAT) or die "Cannot open $ip_file: $!";
flock($fh_ip, LOCK_EX);
while (<$fh_ip>) {
  chomp;
  my ($addr, $t) = split /,/;
  # 指定時間以内のデータのみ保持（古いデータは掃除）
  if ($t > $time - $expire_sec) {
    $last{$addr} = $t;
  }
}

# --- 2. カウント判定 ---
my $should_count = 1;
if (exists $last{$ip} && ($time - $last{$ip}) < $limit_sec) {
  $should_count = 0;
}
$last{$ip} = $time; # 今回の時間をセット

# --- 3. カウンター処理 ---
if (! -e $count_file) {
  open my $init, ">", $count_file; print $init "0"; close $init;
}
open my $fh_cnt, "+<", $count_file or die "Cannot open $count_file";
flock($fh_cnt, LOCK_EX);
my $count = <$fh_cnt> || 0;
$count = int($count);
$count++ if $should_count;
seek($fh_cnt, 0, 0);
print $fh_cnt $count;
truncate($fh_cnt, tell($fh_cnt));
close($fh_cnt);

# --- 4. IP制限ファイルの更新（掃除済みデータを書き戻す） ---
seek($fh_ip, 0, 0);
truncate($fh_ip, 0);
foreach my $addr (keys %last) {
  print $fh_ip "$addr,$last{$addr}\n";
}
close($fh_ip);

# --- 5. ログ保存 ---
open my $fh_log, ">>", $log_file;
flock($fh_log, LOCK_EX);
print $fh_log join(",", scalar localtime(), $ip, $ua, $count, $should_count), "\n";
close $fh_log;

# --- 出力 ---
print "Content-Type: text/plain\n\n";
print $count;
