unit PetitInput;

{$MODE DELPHI}

interface

uses
  Windows, MMSystem, Classes, SysUtils;

const
  // マウスボタン
  MOUSE_LEFT   = 0;
  MOUSE_RIGHT  = 1;
  MOUSE_MIDDLE = 2;

  // ゲームパッド (WinMM用)
  PAD_BUTTON1  = JOY_BUTTON1;
  PAD_BUTTON2  = JOY_BUTTON2;
  PAD_BUTTON3  = JOY_BUTTON3;
  PAD_BUTTON4  = JOY_BUTTON4;
  PAD_BUTTON5  = JOY_BUTTON5;
  PAD_BUTTON6  = JOY_BUTTON6;
  PAD_UP       = $10000; // 十字キー上
  PAD_DOWN     = $20000; // 十字キー下
  PAD_LEFT     = $40000; // 十字キー左
  PAD_RIGHT    = $80000; // 十字キー右

type
  TPetitInput = class
  private
    FhWnd: HWND;
    // キーボード
    FCurKeys, FPrevKeys: array[0..255] of Byte;
    // マウス
    FMouseX, FMouseY: Integer;
    FDeltaX, FDeltaY: Integer;
    FCurMouse, FPrevMouse: Integer;
    FWheelAccum: Integer; // 蓄積用
    FWheelDelta: Integer; // 公開用
    // パッド
    FHasPad: Boolean;
    FCurPad, FPrevPad: DWORD;

  public
    constructor Create(ahWnd: HWND);

    // 毎フレームの最初に必ず呼ぶ
    procedure Update;

    // マウスホイール用：PetitDrawのWindowProc等から呼ばれる必要がある
    procedure AddWheelDelta(zDelta: Integer);

    // --- キー判定 ---
    function IsKeyDown(VK: Integer): Boolean;
    function IsKeyTrigger(VK: Integer): Boolean;
    function IsKeyRelease(VK: Integer): Boolean;

    // --- マウス判定 ---
    function IsMouseDown(Btn: Integer): Boolean;
    function IsMouseTrigger(Btn: Integer): Boolean;
    function IsMouseIn(x, y, w, h: Integer): Boolean;

    // --- パッド判定 ---
    function IsPadDown(Mask: DWORD): Boolean;
    function IsPadTrigger(Mask: DWORD): Boolean;

    // --- プロパティ ---
    property MouseX: Integer read FMouseX;
    property MouseY: Integer read FMouseY;
    property DeltaX: Integer read FDeltaX; // 前フレームからの移動量X
    property DeltaY: Integer read FDeltaY; // 前フレームからの移動量Y
    property Wheel:  Integer read FWheelDelta; // 前フレームからのホイール回転量
    property HasPad: Boolean read FHasPad;
    property Handle: HWND    read FhWnd;
  end;

implementation

constructor TPetitInput.Create(ahWnd: HWND);
begin
  FhWnd := ahWnd;
  FHasPad := joyGetNumDevs() > 0;
  FWheelAccum := 0;
  FWheelDelta := 0;
end;

procedure TPetitInput.AddWheelDelta(zDelta: Integer);
begin
  // zDelta は通常 120 単位で送られてくる
  Inc(FWheelAccum, zDelta);
end;

procedure TPetitInput.Update;
var
  P: TPoint;
  ji: TJoyInfoEx;
begin
  // --- 0. 未接続なら定期的に再検出を試みる ---
  if not FHasPad then
  begin
    // ji を初期化してから確認を試みる
    FillChar(ji, SizeOf(ji), 0);
    ji.dwSize := SizeOf(TJoyInfoEx);
    FHasPad := joyGetPosEx(JOYSTICKID1, @ji) = JOYERR_NOERROR;
  end;

  // 1. キーボード更新
  FPrevKeys := FCurKeys;
  GetKeyboardState(FCurKeys);

  // 2. マウスボタン更新
  FPrevMouse := FCurMouse;
  FCurMouse := 0;
  if (GetKeyState(VK_LBUTTON) and $8000) <> 0 then FCurMouse := FCurMouse or (1 shl MOUSE_LEFT);
  if (GetKeyState(VK_RBUTTON) and $8000) <> 0 then FCurMouse := FCurMouse or (1 shl MOUSE_RIGHT);
  if (GetKeyState(VK_MBUTTON) and $8000) <> 0 then FCurMouse := FCurMouse or (1 shl MOUSE_MIDDLE);

  // 3. マウス座標と移動量(Delta)更新
  GetCursorPos(P);
  Windows.ScreenToClient(FhWnd, P);
  FDeltaX := P.X - FMouseX;
  FDeltaY := P.Y - FMouseY;
  FMouseX := P.X;
  FMouseY := P.Y;

  // 4. ホイール値の確定とリセット
  FWheelDelta := FWheelAccum;
  FWheelAccum := 0;

  // 5. ゲームパッド更新
  if FHasPad then
  begin
    FPrevPad := FCurPad;
    FillChar(ji, SizeOf(ji), 0);
    ji.dwSize := SizeOf(TJoyInfoEx);
    ji.dwFlags := JOY_RETURNALL or JOY_RETURNPOV;

    if joyGetPosEx(JOYSTICKID1, @ji) = JOYERR_NOERROR then
    begin
      FCurPad := ji.dwButtonNumber;
      if ji.wXpos < 16384 then FCurPad := FCurPad or PAD_LEFT;
      if ji.wXpos > 49152 then FCurPad := FCurPad or PAD_RIGHT;
      if ji.wYpos < 16384 then FCurPad := FCurPad or PAD_UP;
      if ji.wYpos > 49152 then FCurPad := FCurPad or PAD_DOWN;
    end
    else
    begin
      // 抜かれたら HasPad を False に戻す
      FHasPad := False;
      FCurPad := 0;
    end;
  end;
end;
{ --- 判定メソッド群 --- }

function TPetitInput.IsKeyDown(VK: Integer): Boolean;
begin
  Result := (FCurKeys[VK] and $80) <> 0;
end;

function TPetitInput.IsKeyTrigger(VK: Integer): Boolean;
begin
  Result := ((FCurKeys[VK] and $80) <> 0) and ((FPrevKeys[VK] and $80) = 0);
end;

function TPetitInput.IsKeyRelease(VK: Integer): Boolean;
begin
  Result := ((FCurKeys[VK] and $80) = 0) and ((FPrevKeys[VK] and $80) <> 0);
end;

function TPetitInput.IsMouseDown(Btn: Integer): Boolean;
begin
  Result := (FCurMouse and (1 shl Btn)) <> 0;
end;

function TPetitInput.IsMouseTrigger(Btn: Integer): Boolean;
begin
  Result := ((FCurMouse and (1 shl Btn)) <> 0) and ((FPrevMouse and (1 shl Btn)) = 0);
end;

function TPetitInput.IsMouseIn(x, y, w, h: Integer): Boolean;
begin
  Result := (FMouseX >= x) and (FMouseX < x + w) and (FMouseY >= y) and (FMouseY < y + h);
end;

function TPetitInput.IsPadDown(Mask: DWORD): Boolean;
begin
  Result := (FCurPad and Mask) <> 0;
end;

function TPetitInput.IsPadTrigger(Mask: DWORD): Boolean;
begin
  Result := ((FCurPad and Mask) <> 0) and ((FPrevPad and Mask) = 0);
end;

end.
