unit PetitDraw;

{$MODE DELPHI}

interface

uses
  Windows, Messages, SysUtils, Classes;

const
  COL_NONE = $FFFFFFFF; // 色指定なし（透明）を表す定数

type
  TNotifyEvent = procedure of object;
  TWheelAccumProc = procedure(zDelta: Integer) of object;

  { TPetitDraw: Windows APIを使用した軽量な描画エンジンクラス }
  TPetitDraw = class
  private
    FOnWheelAccum: TWheelAccumProc; // マウスホイールイベント
    FhWnd: HWND;                    // ウィンドウハンドル
    FWidth, FHeight: Integer;       // キャンバスサイズ
    FBackDC: HDC;                   // バックバッファ用デバイスコンテキスト
    FBackBitmap, FOldBitmap: HBITMAP; // ダブルバッファリング用ビットマップ
    FOnUpdate: TNotifyEvent;        // メインループで呼ばれる更新イベント

    procedure CreateBackBuffer;     // 描画用メモリ領域の確保
    procedure DestroyBackBuffer;    // メモリ領域の解放
  public
    property Handle: HWND read FhWnd;

    constructor Create(Title: string; Width, Height: Integer);
    destructor Destroy; override;

    { 描画基本操作 }
    procedure Clear(Color: COLORREF); // 画面塗りつぶし
    procedure Present;                // バックバッファの内容を画面に反映
    procedure Run;                    // メインループの開始

    { 図形描画メソッド }
    procedure DrawLine(x1, y1, x2, y2: Integer; Width: Integer; Color: COLORREF);
    procedure DrawCircle(x, y, Radius: Integer; BorderWidth: Integer; BorderColor, FillColor: COLORREF);
    procedure DrawRect(x1, y1, x2, y2: Integer; BorderWidth: Integer; BorderColor, FillColor: COLORREF);
    procedure DrawText(Text: string; x, y, Size: Integer; Color: COLORREF; FontName: string = 'MS UI Gothic');

    { イベントプロパティ }
    property OnUpdate: TNotifyEvent read FOnUpdate write FOnUpdate;
    property OnWheelAccum: TWheelAccumProc read FOnWheelAccum write FOnWheelAccum;
  end;

implementation

const
  CP_UTF8 = 65001;

{ ウィンドウメッセージを処理するコールバック関数 }
function WindowProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM): LRESULT; stdcall;
var
  Canvas: TPetitDraw;
  ps: PAINTSTRUCT;
  dc: HDC;
begin
  // ウィンドウに関連付けられた TPetitDraw インスタンスを取得
  Canvas := TPetitDraw(PtrInt(GetWindowLongPtr(hWnd, GWL_USERDATA)));

  case Msg of
    WM_MOUSEWHEEL: begin
      if (Canvas <> nil) and Assigned(Canvas.FOnWheelAccum) then
        Canvas.FOnWheelAccum(SmallInt(HIWORD(wParam))); // ホイール回転量の通知
      Result := 0;
    end;
    WM_PAINT: begin
      dc := BeginPaint(hWnd, ps);
      // 再描画が必要な際、バックバッファから一気に転送（フリッカー防止）
      if Canvas <> nil then
        BitBlt(dc, 0, 0, Canvas.FWidth, Canvas.FHeight, Canvas.FBackDC, 0, 0, SRCCOPY);
      EndPaint(hWnd, ps);
      Result := 0;
    end;
    WM_DESTROY: begin
      PostQuitMessage(0);
      Result := 0;
    end;
    else Result := DefWindowProcW(hWnd, Msg, wParam, lParam);
  end;
end;

{ --- コンストラクタ / デストラクタ --- }

constructor TPetitDraw.Create(Title: string; Width, Height: Integer);
var
  wc: TWndClassW;
  rect: TRect;
  WTitle: array[0..255] of WideChar;
  WLen, ScreenW, ScreenH, WinX, WinY: Integer;
begin
  FWidth := Width; FHeight := Height;

  // 1. タイトルを UTF-8 から WideChar (Unicode) に変換
  WLen := MultiByteToWideChar(CP_UTF8, 0, PChar(Title), Length(Title), WTitle, 255);
  WTitle[WLen] := #0;

  // 2. ウィンドウクラスの登録（初回のみ）
  if not Windows.GetClassInfoW(hInstance, 'PetitDrawClassW', @wc) then
  begin
    wc := Default(TWndClassW);
    wc.lpfnWndProc := @WindowProc;
    wc.hInstance := hInstance;
    wc.lpszClassName := 'PetitDrawClassW';
    wc.hCursor := LoadCursor(0, IDC_ARROW);
    if Windows.RegisterClassW(wc) = 0 then Exit;
  end;

  // 3. クライアント領域（描画部分）を指定サイズにするためのウィンドウ枠計算
  rect := Classes.Rect(0, 0, Width, Height);
  AdjustWindowRect(rect, WS_OVERLAPPEDWINDOW and not (WS_THICKFRAME or WS_MAXIMIZEBOX), False);

  // 4. ウィンドウを画面中央に配置するための座標計算
  ScreenW := GetSystemMetrics(SM_CXSCREEN);
  ScreenH := GetSystemMetrics(SM_CYSCREEN);
  WinX := (ScreenW - (rect.Right - rect.Left)) div 2;
  WinY := (ScreenH - (rect.Bottom - rect.Top)) div 2;

  // 5. ウィンドウの生成
  FhWnd := CreateWindowExW(0, 'PetitDrawClassW', WTitle,
    WS_OVERLAPPEDWINDOW and not (WS_THICKFRAME or WS_MAXIMIZEBOX),
    WinX, WinY, rect.Right - rect.Left, rect.Bottom - rect.Top,
    0, 0, hInstance, nil);

  if FhWnd <> 0 then
  begin
    // Self (インスタンス) をウィンドウに関連付ける
    SetWindowLongPtr(FhWnd, GWL_USERDATA, PtrInt(Self));
    CreateBackBuffer;
    ShowWindow(FhWnd, SW_SHOWNORMAL);
    UpdateWindow(FhWnd);
  end;
end;

destructor TPetitDraw.Destroy;
begin
  DestroyBackBuffer;
  inherited;
end;

{ --- 描画インフラ --- }

procedure TPetitDraw.CreateBackBuffer;
var screenDC: HDC;
begin
  screenDC := GetDC(FhWnd);
  // メモリ上に画面と互換性のある描画領域を作成（ダブルバッファリング）
  FBackDC := CreateCompatibleDC(screenDC);
  FBackBitmap := CreateCompatibleBitmap(screenDC, FWidth, FHeight);
  FOldBitmap := SelectObject(FBackDC, FBackBitmap);
  ReleaseDC(FhWnd, screenDC);
  SetStretchBltMode(FBackDC, COLORONCOLOR);
end;

procedure TPetitDraw.DestroyBackBuffer;
begin
  if FBackDC <> 0 then
  begin
    if FOldBitmap <> 0 then SelectObject(FBackDC, FOldBitmap);
    if FBackBitmap <> 0 then DeleteObject(FBackBitmap);
    DeleteDC(FBackDC);
  end;
end;

procedure TPetitDraw.Clear(Color: COLORREF);
var hBr: Windows.HBRUSH; r: TRect;
begin
  hBr := CreateSolidBrush(Color);
  r := Classes.Rect(0, 0, FWidth, FHeight);
  FillRect(FBackDC, r, hBr);
  DeleteObject(hBr);
end;

procedure TPetitDraw.Present;
var screenDC: HDC;
begin
  // バックバッファの内容を実際の画面(ウィンドウDC)にコピー
  screenDC := GetDC(FhWnd);
  BitBlt(screenDC, 0, 0, FWidth, FHeight, FBackDC, 0, 0, SRCCOPY);
  ReleaseDC(FhWnd, screenDC);
end;

{ --- 図形描画の実装 --- }

procedure TPetitDraw.DrawLine(x1, y1, x2, y2: Integer; Width: Integer; Color: COLORREF);
var hPen, hOld: Windows.HPEN;
begin
  hPen := CreatePen(PS_SOLID, Width, Color);
  hOld := SelectObject(FBackDC, hPen);
  MoveToEx(FBackDC, x1, y1, nil);
  LineTo(FBackDC, x2, y2);
  SelectObject(FBackDC, hOld);
  DeleteObject(hPen);
end;

procedure TPetitDraw.DrawCircle(x, y, Radius: Integer; BorderWidth: Integer; BorderColor, FillColor: COLORREF);
var hPen, hOldP: Windows.HPEN; hBr, hOldB: Windows.HBRUSH;
begin
  hPen := CreatePen(PS_SOLID, BorderWidth, BorderColor);
  if FillColor = COL_NONE then hBr := GetStockObject(NULL_BRUSH)
  else hBr := CreateSolidBrush(FillColor);

  hOldP := SelectObject(FBackDC, hPen);
  hOldB := SelectObject(FBackDC, hBr);
  Ellipse(FBackDC, x - Radius, y - Radius, x + Radius, y + Radius);
  SelectObject(FBackDC, hOldB);
  SelectObject(FBackDC, hOldP);

  if FillColor <> COL_NONE then DeleteObject(hBr);
  DeleteObject(hPen);
end;

procedure TPetitDraw.DrawRect(x1, y1, x2, y2: Integer; BorderWidth: Integer; BorderColor, FillColor: COLORREF);
var hPen, hOldP: Windows.HPEN; hBr, hOldB: Windows.HBRUSH;
begin
  hPen := CreatePen(PS_SOLID, BorderWidth, BorderColor);
  if FillColor = COL_NONE then hBr := GetStockObject(NULL_BRUSH)
  else hBr := CreateSolidBrush(FillColor);

  hOldP := SelectObject(FBackDC, hPen);
  hOldB := SelectObject(FBackDC, hBr);
  Rectangle(FBackDC, x1, y1, x2, y2);
  SelectObject(FBackDC, hOldB);
  SelectObject(FBackDC, hOldP);

  if FillColor <> COL_NONE then DeleteObject(hBr);
  DeleteObject(hPen);
end;

procedure TPetitDraw.DrawText(Text: string; x, y, Size: Integer; Color: COLORREF; FontName: string = 'MS UI Gothic');
var
  hFnt, hOld: HFONT;
  WText: array[0..1024] of WideChar;
  WLen: Integer;
begin
  WLen := MultiByteToWideChar(CP_UTF8, 0, PChar(Text), Length(Text), WText, 1024);
  WText[WLen] := #0;

  // アンチエイリアスを有効にしてフォントを作成
  hFnt := CreateFont(-Size, 0, 0, 0, FW_BOLD, 0, 0, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS,
                     CLIP_DEFAULT_PRECIS, ANTIALIASED_QUALITY, DEFAULT_PITCH or FF_DONTCARE, PChar(FontName));

  hOld := SelectObject(FBackDC, hFnt);
  SetTextColor(FBackDC, Color);
  SetBkMode(FBackDC, TRANSPARENT);
  TextOutW(FBackDC, x, y, WText, WLen);
  SelectObject(FBackDC, hOld);
  DeleteObject(hFnt);
end;

{ --- メインループ --- }

procedure TPetitDraw.Run;
const
  TARGET_FPS = 60;
  FRAME_TIME = 1000 / TARGET_FPS;
var
  Msg: TMsg;
  NextTick, CurrTick: Double;
  Diff: Integer;
begin
  NextTick := GetTickCount64;
  while True do
  begin
    // 1. Windowsメッセージの処理
    if PeekMessage(Msg, 0, 0, 0, PM_REMOVE) then
    begin
      if Msg.message = WM_QUIT then Break;
      TranslateMessage(Msg);
      DispatchMessage(Msg);
    end
    else begin
      // 2. フレーム更新タイミングの制御
      CurrTick := GetTickCount64;
      if CurrTick >= NextTick then
      begin
        if Assigned(FOnUpdate) then FOnUpdate; // ユーザー定義の更新処理を実行

        NextTick := NextTick + FRAME_TIME;
        // 処理が重くて時間を超過している場合は、現在時刻に合わせる
        if NextTick < CurrTick then NextTick := CurrTick + FRAME_TIME;
      end
      else begin
        // 3. CPU負荷を下げるための待機
        Diff := Round(NextTick - CurrTick);
        if Diff > 1 then Sleep(Diff - 1) else Sleep(0);
      end;
    end;
  end;
end;

end.
