// ===============================
// 設定と定数
// ===============================

let userName = "USER-A";
let lastId = 0;
let isSyncing = false;
let connectionStatus = "ONLINE";
let errorCount = 0;
let audioContext;
let syncTimer;

const PALETTE = {
    BLACK: "#000000", BLUE: "#0000AA", LIGHT_GRAY: "#AAAAAA", 
    DARK_GRAY: "#555555", LIGHT_BLUE: "#5555FF", WHITE: "#FFFFFF",
    SYSTEM_RED: "#FF5555", ADMIN_YELLOW: "#FFFF55", ONLINE_GREEN: "#00FF00"
};

let messages = [];
const maxMessages = 30;
const reload = 30000;
const MAX_INPUT_LEN = 150;

let scrollOffset = 0;
let winW = 600, winH = 450;
let winX = 0, winY = 0; 
let dragging = false, dragOffsetX = 0, dragOffsetY = 0;
let isBooting = true;
let mouseX = 0, mouseY = 0;
let totalHeight = 0;
let hoverIndex = -1;
let isHoverName = false;
let isReady = false;

const canvas = document.getElementById("crt");
const ctx = canvas.getContext("2d");
const input = document.getElementById("chatInput");

// ===============================
// セキュリティ & ユーティリティ
// ===============================

// 特殊文字のエスケープ処理
function escapeHTML(str) {
    if (!str) return "";
    return str.replace(/[&<>"']/g, m => ({
        '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;'
    }[m]));
}

// エスケープされた文字を復元（描画用）
function recoverHTML(str) {
    if (!str) return "";
    const map = {
        '&amp;': '&', '&lt;': '<', '&gt;': '>', '&quot;': '"', '&#39;': "'"
    };
    return str.replace(/&amp;|&lt;|&gt;|&quot;|&#39;/g, m => map[m]);
}

// 矩形波によるシステムビープ音の再生
function playBeep(freq, dur) {
    if (!audioContext) {
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
    }
    const osc = audioContext.createOscillator();
    const gain = audioContext.createGain();
    osc.type = 'square';
    osc.frequency.setValueAtTime(freq, audioContext.currentTime);
    gain.gain.setValueAtTime(0.05, audioContext.currentTime);
    gain.gain.exponentialRampToValueAtTime(0.0001, audioContext.currentTime + dur);
    osc.connect(gain); gain.connect(audioContext.destination);
    osc.start(); osc.stop(audioContext.currentTime + dur);
}

// ===============================
// システムロジック
// ===============================

// 画面中央にウィンドウを配置
function centerWindow() {
    winX = (canvas.width - winW) / 2;
    winY = (canvas.height - (winH + 80)) / 2;
    updateInputPosition();
}

// HTML入力フォームの位置をキャンバス上のウィンドウに同期
function updateInputPosition() {
    input.style.left = winX + "px";
    input.style.top = (winY + winH + 30) + "px"; 
    input.style.width = (winW - 28) + "px"; 
    input.style.transform = "none";
}

// リサイズイベント
function resize() {
    canvas.width = window.innerWidth;
    canvas.height = window.innerHeight;
    centerWindow();
}

window.addEventListener('resize', resize);
resize();

// 入室通知の送信
async function sendSystemNotification(actionType) {
    const msg = actionType === 'JOIN' ? 
        `${userName} HAS CONNECTED.` : 
        `${userName} HAS DISCONNECTED.`;

    const params = new URLSearchParams();
    params.append('action', 'send');
    params.append('name', 'SYSTEM');
    params.append('message', msg);

    try {
        const response = await fetch('../cgi/ct68_cgi.cgi', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: params.toString()
        });
        
        const result = await response.text();
        if (result.includes("ERROR")) {
            console.error("Server Error:", result);
        } else {
            await syncChat();
        }
    } catch (e) {}
}

// コマンドパターン定義
const COMMAND_PATTERNS = [
    { keywords: ["HELLO"], response: "システムは正常稼働中です。" },
    { keywords: ["DATE", "TIME"], response: () => `現在は ${new Date().toLocaleString()} です。` },
    { keywords: ["HELP"], response: "利用可能: HELLO, DATE, TIME, HELP, VERSION, CLEAR" },
    { keywords: ["VERSION"], response: "Cyber-Terminal 68 Ver 1.00 Build 2026" },
    { 
        keywords: ["CLEAR"], 
        action: () => {
            addMessage("SYSTEM > INITIATING MEMORY WIPE...");
            playBeep(440, 0.2);
            setTimeout(() => {
                messages = [];
                scrollOffset = 0;
                addMessage("SYSTEM > MEMORY CLEANSED.");
            }, 800);
        }
    }
];

// 入力文字列をコマンドとして処理
function processCommand(inputTxt) {
    const upperInput = inputTxt.toUpperCase().trim();
    for (const item of COMMAND_PATTERNS) {
        if (item.keywords.some(key => upperInput.includes(key))) {
            if (item.action) item.action();
            if (item.response) {
                const res = (typeof item.response === 'function') ? item.response() : item.response;
                addMessage(`SYSTEM > ${res}`);
                playBeep(1200, 0.05);
            }
            return true;
        }
    }
    return false;
}

// 起動シーケンス
async function startBootSequence() {
    const bootLogs = [
        "MEMORY CHECK OK... 12288KB",
        "IPL ROM VERSION 1.0 (C)2026",
        "Powered by Clamini Library",
        "LOADING SYSTEM FILES...",
        "SYSTEM READY."
    ];
    for (let log of bootLogs) {
        addMessage(log);
        playBeep(1000, 0.03);
        await new Promise(r => setTimeout(r, 400));
    }

    // 1. 過去ログを既読にする
    await syncChat(true); 

    // 2. 準備完了フラグを立てる
    isReady = true; 

    // 3. 入室通知を送信（awaitを付けることで確実に送信を完了させる）
    await sendSystemNotification('JOIN'); 

    isBooting = false;
    input.placeholder = "Enter command...";
    addMessage(`SYSTEM > Hello, ${userName}.`);

    // 4. 定期同期を開始
    startPeriodicSync();
}

// 同期管理

function startPeriodicSync() {
    if (syncTimer) clearInterval(syncTimer);
    syncTimer = setInterval(() => {
        if (!isBooting) syncChat();
    }, reload);
}

// メッセージの構造化と追加
function addMessage(text) {
    const safeText = recoverHTML(text);
    const separator = " > ";
    const parts = safeText.split(separator);
    const namePart = parts.length > 1 ? parts[0] + separator : "";
    const bodyPart = parts.length > 1 ? parts[1] : parts[0];

    ctx.font = "16px monospace";
    const nameWidth = ctx.measureText(namePart).width;
    const bodyLines = wrapText(bodyPart, winW - 60 - nameWidth);

    messages.push({
        namePart, nameWidth, bodyPart, bodyLines,
        fullTextLength: safeText.length,
        currentIndex: 0, frame: 0, delay: 1, flashFrame: 0
    });
    if (messages.length > maxMessages) messages.shift();
    updateScroll();
}

// テキストの自動折り返し計算
function wrapText(text, maxWidth) {
    const words = text.split("");
    let lines = [], currentLine = "";
    for (let char of words) {
        let testLine = currentLine + char;
        if (ctx.measureText(testLine).width > maxWidth) {
            lines.push(currentLine);
            currentLine = char;
        } else { currentLine = testLine; }
    }
    lines.push(currentLine);
    return lines;
}

// 全メッセージの合計高さを計算し、自動スクロール位置を更新
function updateScroll() {
    totalHeight = 0;
    messages.forEach(msg => { totalHeight += (msg.bodyLines.length * 20) + 2; });
    const viewHeight = winH - 60;
    scrollOffset = Math.max(0, totalHeight - viewHeight);
}

// サーバー同期 (CGI連携)
async function syncChat(silent = false) {
    if (isSyncing) return;
    isSyncing = true;
    connectionStatus = "SYNCING";

    try {
        const response = await fetch(`../cgi/ct68_cgi.cgi?since=${lastId}`);
        if (!response.ok) throw new Error(`HTTP_${response.status}`);
        
        const rawData = await response.text();
        connectionStatus = "ONLINE";
        errorCount = 0;

        if (rawData.trim() && rawData !== "OK") {
            const lines = rawData.trim().split('\n');
            lines.forEach(line => {
                const parts = line.split(',');
                if (parts.length >= 3) {
                    const id = parseInt(parts[0]);
                    const name = parts[1];
                    const msg = parts[2];

                    if (id > lastId) {
                        // silent フラグが false の時だけ画面に表示
                        if (!silent) {
                            addMessage(`${name.toUpperCase()} > ${msg}`);
                            playBeep(660, 0.05);
                        }
                        lastId = id; 
                    }
                }
            });
        }
    } catch (e) {
        errorCount++;
        connectionStatus = "OFFLINE";
        if (errorCount === 1) {
            addMessage("SYSTEM > WARNING: LINE DISCONNECTED. RETRYING...");
            playBeep(200, 0.3);
        }
    } finally {
        isSyncing = false;
    }
}

// ===============================
// イベントリスナー
// ===============================

// チャット送信イベント
input.addEventListener("keydown", async (e) => {
    if (isBooting || e.key !== "Enter") return;
    const txt = input.value.trim();
    if (!txt) return;

    if (txt.length > MAX_INPUT_LEN) {
        addMessage("SYSTEM > ERROR: BUFFER OVERFLOW");
        return;
    }

    // コマンドとして処理された場合は、サーバー送信を行わずに終了する
    if (processCommand(txt)) {
        input.value = ""; // 入力欄をクリア
        return; 
    }

    input.value = "";
    input.disabled = true;

    // パラメータの構築（統一した書き方）
    const params = new URLSearchParams();
    params.append('action', 'send');
    params.append('name', userName);
    params.append('message', txt);

    try {
        const response = await fetch('../cgi/ct68_cgi.cgi', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: params.toString()
        });

        if (!response.ok) throw new Error();
        
        const result = await response.text();
        if (result.includes("ERROR")) {
            throw new Error(result);
        }

        await syncChat(); // 送信直後に差分同期
    } catch (err) {
        addMessage("SYSTEM > ERROR: PACKET LOST. RE-ENTRY REQUIRED.");
        input.value = txt; // 失敗時に文字を戻す
        playBeep(150, 0.3);
        console.error("Send Error:", err);
    } finally {
        input.disabled = false;
        input.focus();
    }
});

// スクロール操作
canvas.addEventListener("wheel", (e) => {
    scrollOffset += e.deltaY;
    const viewHeight = winH - 60;
    const maxScroll = Math.max(0, totalHeight - viewHeight);
    scrollOffset = Math.min(Math.max(0, scrollOffset), maxScroll);
    e.preventDefault();
}, { passive: false });

// ドラッグ開始
canvas.addEventListener("mousedown", (e) => {
    const mx = e.clientX - canvas.getBoundingClientRect().left;
    const my = e.clientY - canvas.getBoundingClientRect().top;
    if (mx >= winX && mx <= winX + winW && my >= winY && my <= winY + 24) {
        dragging = true;
        dragOffsetX = mx - winX; dragOffsetY = my - winY;
    }
});

// マウス移動（ドラッグとホバー判定）
window.addEventListener("mousemove", (e) => {
    const rect = canvas.getBoundingClientRect();
    mouseX = e.clientX - rect.left;
    mouseY = e.clientY - rect.top;

    if (dragging) {
        let nextX = mouseX - dragOffsetX;
        let nextY = mouseY - dragOffsetY;
        nextX = Math.max(0, Math.min(nextX, canvas.width - winW));
        nextY = Math.max(0, Math.min(nextY, canvas.height - winH - 80));
        winX = nextX; winY = nextY;
        updateInputPosition();
        return;
    }

    // ホバー判定ロジック
    hoverIndex = -1;
    let checkY = winY + 40 - scrollOffset; // drawの初期値と合わせる

    for (let i = 0; i < messages.length; i++) {
        const msg = messages[i];
        // メッセージ1つ分が占有する全高さ (行数 * 20px + 余白 2px)
        const msgTotalHeight = (msg.bodyLines.length * 20) + 2;

        // マウスが現在のメッセージの高さの範囲内にあるか
        if (mouseY >= checkY && mouseY < checkY + msgTotalHeight) {
            if (mouseX > winX + 20 && mouseX < winX + winW - 20) {
                hoverIndex = i;
                isHoverName = (mouseX < winX + 20 + msg.nameWidth);
                break;
            }
        }
        checkY += msgTotalHeight;
    }
    
    const isOverTitle = (mouseX >= winX && mouseX <= winX + winW && mouseY >= winY && mouseY <= winY + 24);
    canvas.style.cursor = (isOverTitle || (hoverIndex !== -1 && isHoverName)) ? "pointer" : "default";
});

window.addEventListener("mouseup", () => dragging = false);

// クリックイベント（コピー機能とオーディオ再開）
canvas.addEventListener("click", () => {
    if (audioContext && audioContext.state === 'suspended') {
        audioContext.resume();
    }
    if (hoverIndex !== -1 && isHoverName) {
        const msg = messages[hoverIndex];
        navigator.clipboard.writeText(msg.bodyPart).then(() => {
            msg.flashFrame = 20; // 成功時に白く光らせる
            playBeep(1200, 0.05);
        });
    }
});

// ログインボタン
document.getElementById("loginBtn").addEventListener("click", async () => {
    const val = document.getElementById("userNameInput").value.trim();
    if (val) { userName = val; }
    document.getElementById("loginOverlay").classList.add("hidden");
    playBeep(1200, 0.1);
    await startBootSequence();
});

// 退室通知
window.addEventListener('visibilitychange', () => {
    // 完全に起動が終わっていない、またはログインしていない時は無視
    if (!isReady || isBooting) return;

    if (document.visibilityState === 'hidden') {
        const url = '../cgi/ct68_cgi.cgi';
        const data = new URLSearchParams({
            action: 'send',
            name: 'SYSTEM',
            message: `${userName} HAS DISCONNECTED.`
        });
        navigator.sendBeacon(url, data);
    }
});

document.getElementById("userNameInput").addEventListener("keydown", (e) => {
    if (e.key === "Enter") document.getElementById("loginBtn").click();
});

// ===============================
// 描画ループ
// ===============================

function draw() {
    ctx.fillStyle = "#7A7A7A";
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    // ウィンドウ影と本体
    ctx.fillStyle = "rgba(0,0,0,0.4)";
    ctx.fillRect(winX + 10, winY + 10, winW, winH);
    ctx.fillStyle = PALETTE.DARK_GRAY;
    ctx.fillRect(winX, winY, winW, winH);
    ctx.strokeStyle = PALETTE.WHITE;
    ctx.strokeRect(winX, winY, winW, winH);
    ctx.fillStyle = PALETTE.BLACK;
    ctx.fillRect(winX + 4, winY + 24, winW - 8, winH - 28);

    // タイトルバー
    ctx.fillStyle = dragging ? PALETTE.LIGHT_BLUE : PALETTE.BLUE;
    ctx.fillRect(winX + 4, winY + 4, winW - 8, 20);

    // ステータスランプ（接続状況を可視化）
    let statusColor = PALETTE.ONLINE_GREEN;
    if (connectionStatus === "SYNCING") statusColor = PALETTE.ADMIN_YELLOW;
    if (connectionStatus === "OFFLINE") statusColor = PALETTE.SYSTEM_RED;
    ctx.fillStyle = statusColor;
    ctx.beginPath();
    ctx.arc(winX + winW - 15, winY + 14, 5, 0, Math.PI * 2);
    ctx.fill();

    ctx.fillStyle = PALETTE.WHITE;
    ctx.font = "14px monospace";
    ctx.fillText(`Cyber-Terminal 68 --- Login USER : ${userName}`, winX + 12, winY + 18);

    // メッセージ描画開始
    ctx.save();
    ctx.beginPath();
    ctx.rect(winX + 4, winY + 24, winW - 8, winH - 28);
    ctx.clip();
    
    ctx.font = "16px monospace";
    ctx.textBaseline = "top";
    let currentY = winY + 40 - scrollOffset;
    
    messages.forEach((msg, idx) => {
        const msgH = (msg.bodyLines.length * 20) + 2;

        // 画面外（上下）のメッセージは描画をスキップして負荷軽減
        if (currentY + msgH > winY + 24 && currentY < winY + winH) {
            
            // 1. タイピング進捗の計算
            const charLimit = msg.currentIndex;
            const nameLimit = Math.min(charLimit, msg.namePart.length);
            const bodyLimit = Math.max(0, charLimit - msg.namePart.length);

            // 2. 名前の色設定と描画
            let nameColor = msg.namePart.includes("SYSTEM") ? PALETTE.SYSTEM_RED : 
                            (idx === hoverIndex && isHoverName) ? PALETTE.ONLINE_GREEN : PALETTE.LIGHT_GRAY;
            
            // コピー時のフラッシュ演出
            if (msg.flashFrame > 0) {
                nameColor = PALETTE.WHITE;
                msg.flashFrame--;
            }
            
            ctx.fillStyle = nameColor;
            ctx.fillText(msg.namePart.slice(0, nameLimit), winX + 20, currentY);

            // 3. 本文の色設定と複数行描画
            ctx.fillStyle = PALETTE.LIGHT_GRAY;
            let processedChars = 0;
            msg.bodyLines.forEach((line, li) => {
                const remainingInBody = bodyLimit - processedChars;
                if (remainingInBody > 0) {
                    const lineToShow = line.slice(0, remainingInBody);
                    // 名前ラベルの横幅分ずらして描画
                    ctx.fillText(lineToShow, winX + 20 + msg.nameWidth, currentY + (li * 20));
                }
                processedChars += line.length;
            });
        }
        // 次のメッセージの開始位置を更新
        currentY += msgH;
    });
    ctx.restore();

    // 走査線
    ctx.fillStyle = "rgba(0, 0, 0, 0.05)";
    for (let i = 0; i < canvas.height; i += 4) ctx.fillRect(0, i, canvas.width, 1);
    
        // 入力文字数カウンター
    if (!isBooting) {
        ctx.fillStyle = input.value.length >= MAX_INPUT_LEN ? PALETTE.SYSTEM_RED : PALETTE.WHITE;
        ctx.font = "12px monospace";
        ctx.fillText(`LEN:${input.value.length}/${MAX_INPUT_LEN}`, winX + 10, winY + winH + 15);
    }
}

function loop() {
    messages.forEach(m => { if(m.currentIndex < m.fullTextLength) m.currentIndex++; });
    draw();
    requestAnimationFrame(loop);
}

// 実行開始
loop();
